/*
 * Routines for communicating with FMS.
 */
#include <time.h>
#ifndef _WIN32
#include <unistd.h>
#include <netinet/in.h>
#else
#include <winsock2.h>
#endif

#include "libfma.h"
#include "lf_internal.h"
#include "lf_internal.h"
#include "lf_alert.h"
#include "lf_fms_comm.h"
#include "lf_fms.h"

/*
 * Open a connection to the FMS
 */
int
lf_fms_connect(
  char *fms_names)
{
  char *w[LF_STRING_LEN];
  int nh;
  int h;

  /* If name list specified, use it */
  if (fms_names != NULL) {
    line2words(fms_names, w, ",", 0, &nh);

  } else {

    /* next, check for environment variable */
    fms_names = getenv("FMS_SERVER");
    if (fms_names != NULL) {
      line2words(fms_names, w, ",", 0, &nh);

    /* not specified and not in environment, use compiled default */
    } else {
      w[0] = Lf_dflt_fms_server;
      nh = 1;
    }
  }

  /* Loop through list of hosts trying to connect */
  for (h=0; h<nh; ++h) {
    int sock;
    uint8_t conn_type;
    int rc;

    sock = lf_connect_to_host(w[h], FMS_SERVICE_PORT);
    if (sock == -1) continue;

    conn_type = FMS_CONN_GENERIC;
    rc = lf_write(sock, &conn_type, sizeof(conn_type));
    if (rc != sizeof(conn_type)) {
      close(sock);
      continue;
    }

    return sock;
  }

  LF_ERROR(("Cannot connect to FMS"));
 except:
  return -1;
}

/*
 * Close a connection to FMS
 */
void
lf_fms_disconnect(
  int fms)
{
  close(fms);
}

/*
 * Send a message header to FMS
 */
int
lf_send_fms_header(
  int fms,
  int type,
  int len)
{
  struct fms_msg_header hdr;
  int rc;

  /* fill in the message header */
  hdr.length_32 = htonl(len);
  hdr.msg_type_32 = htonl(type);
  
  /* send the header */
  rc = lf_write(fms, &hdr, sizeof(hdr));
  if (rc != sizeof(hdr)) LF_ERROR(("Error sending FMS msg header"));
  return 0;

 except:
  return -1;
}

/*
 * Send a message to the FMS
 */
int
lf_send_fms_msg(
  int fms,
  int type,
  void *msgbuf,
  int len)
{
  int rc;

  /* send a header */
  rc = lf_send_fms_header(fms, type, len);
  if (rc == -1) return -1;

  /* then send the message body */
  rc = lf_write(fms, msgbuf, len);
  if (rc != len) LF_ERROR(("Error sending message to FMS"));
  return 0;

 except:
  return -1;
}

/*
 * Query alerts from the FMS
 */
int
lf_fms_query_alerts(
  int fms,
  int show_acked,
  int show_relics,
  struct lf_alert **app)
{
  struct fms_query_alerts req;
  struct lf_alert *alerts;
  uint32_t val;
  int num_alerts;
  int rc;
  int i;

  alerts = NULL;

  /* build the request (bytes so no flipping) */
  req.show_acked = show_acked;
  req.show_relics = show_relics;

  /* send the message */
  rc = lf_send_fms_msg(fms, LF_CLIENT_FMS_QUERY_ALERTS, &req, sizeof(req));
  if (rc != 0) return -1;

  /* now read number of alerts to expect */
  rc = lf_read(fms, &val, sizeof(val));
  if (rc != sizeof(val)) LF_ERROR(("Error reading number of alerts"));
  num_alerts = ntohl(val);

  /* allocate space for the alerts */
  LF_CALLOC(alerts, struct lf_alert, num_alerts);

  /* read in each one */
  for (i=0; i<num_alerts; ++i) {
    rc = lf_read(fms, alerts+i, sizeof(*alerts));
    if  (rc != sizeof(*alerts)) LF_ERROR(("Error reading alert from FMS"));

    lf_alert_ntoh(alerts+i);	/* convert from network order */
  }

  /* return number of alerts and array of alerts */
  *app = alerts;
  return num_alerts;

 except:
  LF_FREE(alerts);
  return -1;
}

/*
 * Acknowledge an alert to the FMS
 */
int
lf_ack_alert(
  int fms,
  uint32_t alert_id)
{
  struct fms_ack_alert req;
  int rc;

  /* fill in message struct */
  req.alert_id_32 = htonl(alert_id);

  /* send the message */
  rc = lf_send_fms_msg(fms, LF_CLIENT_FMS_ACK_ALERT, &req, sizeof(req));
  if (rc != 0) return -1;

  return 0;
}

/*
 * Get current status from the FMS
 */
int
lf_fms_query_status(
  int fms,
  struct lf_fms_status *sp)
{
  struct lf_fms_status_msg msg;
  struct fms_query_status req;
  int rc;

  /* send the message */
  rc = lf_send_fms_msg(fms, LF_CLIENT_FMS_QUERY_STATUS, &req, sizeof(req));
  if (rc != 0) return -1;

  /* Get the response message */
  rc = lf_read(fms, &msg, sizeof(msg));
  if (rc != sizeof(msg)) {
    LF_ERROR(("Error reading status from FMS"));
  }

  /* copy out the status */
  sp->hosts_known = ntohl(msg.hosts_known_32);
  sp->fmas_found = ntohl(msg.fmas_found_32);
  sp->unacked_alerts = ntohl(msg.unacked_alerts_32);
  sp->mapping_complete = msg.mapping_complete_8;
  strcpy(sp->last_mapper, msg.last_mapper);
  sp->db_complete = msg.db_complete_8;

  return 0;

 except:
  return -1;
}

/*
 * Send a switch maintenance message to FMS
 */
int
lf_fms_switch_maintenance(
  int fms,
  char *enc_name,
  int slot,
  int port_label,
  int up)
{
  struct fms_switch_maintenance msg;
  struct lf_fms_return_code reply;
  int rc;

  /* build the maintenance request message */
  strcpy(msg.enc_name, enc_name);
  msg.slot = slot;
  msg.port_label = port_label;
  msg.up = up;

  /* send the message */
  rc = lf_send_fms_msg(fms, LF_CLIENT_FMS_SWITCH_MAINTENANCE,
      &msg, sizeof(msg));
  if (rc != 0) return -1;

  /* Get the response message */
  rc = lf_read(fms, &reply, sizeof(reply));
  if (rc != sizeof(reply)) {
    LF_ERROR(("Error reading return code from FMS"));
  }

  /* look at the return code */
  rc = ntohl(reply.fms_rc_32);
  if (rc != 0) {
    LF_ERROR(("Error setting switch maintenance mode, reason=\"%s\"",
	  reply.reason));
  }

  return 0;

 except:
  return -1;
}
